local data_util = require("data-util")
local flib_table = require("__flib__.table")

local autogenerated = false
local function check_autogenerated()
  if autogenerated then
    -- We want this to always crash.
    error(
      "Flare stack recipes have already been generated. Use flare_stack_lib.make_recipe() to create recipes manually."
    )
  end
end

--- Configures the behavior of the Krastorio 2 flare stack. At the beginning of the data-updates phase, Krastorio will
--- auto-generate flare stack recipes for all fluids in the game except for those in the blacklist. If a mod needs to
--- add a flare stack recipe after the auto-generation has been completed, they can do so by calling `make_recipe`
--- manually.
--- @class k2.flare_stack_lib
local flare_stack_lib = {}

--- Fluids in this blacklist will not have auto-generated flare stack recipes.
--- @type table<data.FluidID, boolean>
local auto_blacklist = {
  ["kr-matter"] = true,
}

--- Adds the given fluid to the auto-generation blacklist.
--- @param fluid_name data.FluidID
function flare_stack_lib.add_blacklist(fluid_name)
  check_autogenerated()
  auto_blacklist[fluid_name] = true
end

--- Removes the given fluid from the auto-generation blacklist.
--- @param fluid_name data.FluidID
function flare_stack_lib.remove_blacklist(fluid_name)
  check_autogenerated()
  auto_blacklist[fluid_name] = nil
end

--- Returns whether the given fluid is in the auto-generation blacklist.
--- @param fluid_name data.FluidID
--- @return boolean
function flare_stack_lib.is_blacklisted(fluid_name)
  return auto_blacklist[fluid_name] and true or false
end

--- Specifies byproducts that are to be created when burning the given fluid.
--- @type table<data.FluidID, data.ProductPrototype[]>
local auto_byproducts = {
  ["kr-dirty-water"] = { { type = "item", name = "stone", amount = 1, probability = 0.3 } },
}

--- Adds a flare stack byproduct for the given fluid.
--- @param fluid_name data.FluidID
--- @param byproduct data.ProductPrototype
function flare_stack_lib.add_byproduct(fluid_name, byproduct)
  check_autogenerated()
  table.insert(flib_table.get_or_insert(auto_byproducts, fluid_name, {}), byproduct)
end

--- Sets the flare stack byproducts for the given fluid. Pass `nil` to clear the byproducts.
--- @param fluid_name data.FluidID
--- @param byproducts data.ProductPrototype[]|nil
function flare_stack_lib.set_byproducts(fluid_name, byproducts)
  check_autogenerated()
  auto_byproducts[fluid_name] = byproducts
end

--- Returns the flare stack byproducts for the given fluid.
--- @return data.ProductPrototype[]?
function flare_stack_lib.get_byproducts(fluid_name)
  return auto_byproducts[fluid_name]
end

--- Specifies a custom emissions multiplier for each fluid.
--- @type table<data.FluidID, double>
local auto_fluid_emissions_multiplier = {
  ["kr-ammonia"] = 4.0,
  ["kr-chlorine"] = 2.0,
  ["crude-oil"] = 9.0,
  ["kr-dirty-water"] = 6.0,
  ["heavy-oil"] = 4.0,
  ["kr-hydrogen"] = 0.0,
  ["kr-hydrogen-chloride"] = 2.0,
  ["light-oil"] = 3.0,
  ["lubricant"] = 4.0,
  ["kr-nitrogen"] = 0.0,
  ["kr-oxygen"] = 0.0,
  ["petroleum-gas"] = 2.0,
  ["steam"] = 0.0,
  ["water"] = 0.0,
}

--- Sets the flare stack emissions multiplier for the given fluid. Pass `nil` to clear the multiplier.
--- @param fluid_name data.FluidID
--- @param multiplier double|nil
function flare_stack_lib.set_fluid_emissions_multiplier(fluid_name, multiplier)
  check_autogenerated()
  auto_fluid_emissions_multiplier[fluid_name] = multiplier
end

--- Sets the flare stack emissions multiplier for the given fluid.
--- @param fluid_name data.FluidID
--- @return double?
function flare_stack_lib.get_fluid_emissions_multiplier(fluid_name)
  return auto_fluid_emissions_multiplier[fluid_name]
end

--- @param color data.Color
--- @param alpha double
local function with_alpha(color, alpha)
  return {
    r = color.r or color[1],
    g = color.g or color[2],
    b = color.b or color[3],
    a = alpha,
  }
end

--- Generates a burning recipe for the given fluid.
--- @param fluid_name data.FluidID
--- @param fluid_emissions_multiplier double?
--- @param byproducts data.ProductPrototype[]?
function flare_stack_lib.make_recipe(fluid_name, fluid_emissions_multiplier, byproducts)
  local fluid = data.raw.fluid[fluid_name]
  if not fluid then
    data_util.error("Fluid " .. fluid_name .. " does not exist.")
    return
  end
  if data.raw.recipe["kr-burn-" .. fluid_name] then
    data_util.error("Flare stack recipe for " .. fluid_name .. " already exists.")
    return
  end

  data:extend({
    {
      type = "recipe",
      name = "kr-burn-" .. fluid_name,
      localised_name = { "recipe-name.kr-burn", fluid.localised_name or { "fluid-name." .. fluid_name } },
      icons = flib_table.array_merge({
        {
          { icon = "__Krastorio2Assets__/icons/burn-recipes-background/burn-recipe-corner.png" },
          {
            icon = "__Krastorio2Assets__/icons/burn-recipes-background/burn-recipe-corner-mask.png",
            tint = with_alpha(fluid.base_color, 0.9),
          },
        },
        data_util.transform_icons(data_util.get_icons(fluid), { scale = 0.34 }),
      }),
      subgroup = "kr-flare-stack",
      order = fluid.order,
      hide_from_player_crafting = true,
      enabled = false,
      category = "kr-fuel-burning",
      energy_required = 2,
      ingredients = {
        { type = "fluid", name = fluid_name, amount = 100 },
      },
      results = byproducts,
      emissions_multiplier = fluid_emissions_multiplier,
      always_show_products = byproducts and true or false,
      show_amount_in_title = false,
      crafting_machine_tint = {
        primary = fluid.base_color,
        secondary = with_alpha(fluid.base_color, 0.35),
        tertiary = with_alpha(fluid.flow_color, 0.5),
        quaternary = with_alpha(fluid.flow_color, 0.75),
      },
    },
  })

  data_util.add_recipe_unlock("kr-fluid-excess-handling", "kr-burn-" .. fluid_name)
end

--- Returns the flare stack recipe for the given fluid.
--- @param fluid_name data.FluidID
--- @return data.RecipePrototype?
function flare_stack_lib.get_recipe(fluid_name)
  return data.raw.recipe["kr-burn-" .. fluid_name]
end

--- Auto-generates flare stack recipes. This function MUST NOT be called from other mods; it is used internally by Krastorio.
function flare_stack_lib.auto_generate()
  autogenerated = true
  for fluid_name, fluid in pairs(data.raw.fluid) do
    if
      not auto_blacklist[fluid_name]
      and not fluid.hidden
      and not fluid.parameter
      and not flare_stack_lib.get_recipe(fluid_name)
    then
      flare_stack_lib.make_recipe(fluid_name, auto_fluid_emissions_multiplier[fluid_name], auto_byproducts[fluid_name])
    end
  end
end

return flare_stack_lib
